import { supabase } from '@/config/supabase';
import {
  KYCSubmission,
  KYCStatus,
  KYCReviewAction,
  SubmitKYCRequest
} from './types';

export const kycService = {
  // Submit or update KYC
  async submitKYC(vendor_id: string, req: SubmitKYCRequest) {
    // Upsert by vendor_id
    const { data, error } = await supabase.from('kyc_submissions').upsert({
      vendor_id,
      business_name: req.business_name,
      registration_type: req.registration_type,
      id_type: req.id_type,
      id_number: req.id_number,
      id_document_url: req.id_document_url,
      business_document_url: req.business_document_url || null,
      status: 'pending_review',
      rejection_reason: null,
      submitted_at: new Date().toISOString(),
      reviewed_at: null
    }, { onConflict: 'vendor_id' }).select().single();
    return { data, error };
  },

  // Get KYC status for vendor
  async getKYCStatus(vendor_id: string) {
    const { data, error } = await supabase.from('kyc_submissions').select('status,rejection_reason,submitted_at').eq('vendor_id', vendor_id).single();
    return { data, error };
  },

  // Admin: list pending KYC submissions
  async listPendingKYC(limit = 20, offset = 0) {
    const { data, error } = await supabase.from('kyc_submissions').select('*,vendor:vendor_id(id,business_name,kyc_status)').eq('status', 'pending_review').range(offset, offset + limit - 1).order('submitted_at', { ascending: true });
    return { data, error };
  },

  // Admin: review KYC (approve/reject)
  async reviewKYC(vendor_id: string, action: KYCReviewAction, admin_id: string) {
    // 1. Update kyc_submissions
    const status: KYCStatus = action.action === 'approve' ? 'approved' : 'rejected';
    const rejection_reason = action.action === 'reject' ? (action.rejection_reason || 'Not specified') : null;
    const { data: submission, error: kycError } = await supabase.from('kyc_submissions').update({
      status,
      rejection_reason,
      reviewed_at: new Date().toISOString()
    }).eq('vendor_id', vendor_id).select().single();
    if (kycError) return { error: kycError };
    // 2. Update vendors.kyc_status
    await supabase.from('vendors').update({ kyc_status: status }).eq('id', vendor_id);
    // 3. Log admin action
    await supabase.from('admin_actions').insert({
      admin_id,
      action: `${status}_kyc`,
      target_vendor_id: vendor_id,
      details: rejection_reason
    });
    return { data: submission };
  },

  // Enforcement: check if KYC is required for vendor
  async isKYCRequired(vendor_id: string) {
    // 1. Check if vendor has > 5 paid invoices
    const { data: invoices } = await supabase.from('invoices').select('id').eq('vendor_id', vendor_id).eq('status', 'paid');
    if ((invoices?.length || 0) > 5) return true;
    // 2. Check if SUM(transactions.amount) > 500000
    const { data: txnSum } = await supabase.rpc('sum_transactions_for_vendor', { vendor_id });
    if ((txnSum || 0) > 500000) return true;
    return false;
  },

  // Enforcement: check if vendor is KYC approved
  async isKYCApproved(vendor_id: string) {
    const { data } = await supabase.from('kyc_submissions').select('status').eq('vendor_id', vendor_id).single();
    return data?.status === 'approved';
  }
}; 